##### Rozdział 8: reguły asocjacyjne -------------------

## Przykład: identyfikowanie często kupowanych artykułów spożywczych ----
## Etap 2: badanie i przygotowywanie danych ----

# wczytujemy dane do macierzy rozrzedzonej 
library(arules)
groceries <- read.transactions("groceries.csv", sep = ",")
summary(groceries)

# badamy długi format (bez dekodowania), aby zobaczyć identyfikatory artykułów
head(toLongFormat(groceries, decode = FALSE), n = 7)

# przyglądamy się pierwszym pięciu transakcjom
inspect(groceries[1:5])

# badamy częstość występowania artykułów
itemFrequency(groceries[, 1:3])

# kreślimy częstość występowania artykułów
itemFrequencyPlot(groceries, support = 0.1)
itemFrequencyPlot(groceries, topN = 20)

# wizualizacja macierzy rozrzedzonej dla pierwszych pięciu transakcji
image(groceries[1:5])

# wizualizacja losowej próbki 100 transakcji
image(sample(groceries, 100))

## Etap 3: trenowanie modelu na danych ----
library(arules)

# ustawienia domyślne sprawiają, że model nie uczy się żadnych reguł
apriori(groceries)

# ustawiamy lepsze poziomy wsparcia i ufności, aby model nauczył się więcej reguł
groceryrules <- apriori(groceries, parameter = list(support =
                          0.006, confidence = 0.25, minlen = 2))
groceryrules

## Etap 4: ewaluacja modelu ----
# podsumowanie reguł asocjacyjnych dla artykułów spożywczych
summary(groceryrules)

# przyglądamy się pierwszym trzem regułom
inspect(groceryrules[1:3])

## Etap 5: poprawianie działania modelu ----

# sortujemy reguły według dźwigni
inspect(sort(groceryrules, by = "lift")[1:5])

# znajdujemy podzbiory reguł zawierające owoce jagodowe
berryrules <- subset(groceryrules, items %in% "berries")
inspect(berryrules)

# zapisujemy reguły w pliku CSV
write(groceryrules, file = "groceryrules.csv",
      sep = ",", quote = TRUE, row.names = FALSE)

# przekształcamy zbiór reguł w ramkę danych
groceryrules_df <- as(groceryrules, "data.frame")
str(groceryrules_df)

# zwiększamy efektywność modelu, używając algorytmu eclat
# najpierw generujemy częste zestawy artykułów
groceryitemsets_eclat <- eclat(groceries, support = 0.006)
inspect(groceryitemsets_eclat[1:5])
# następnie generujemy reguły
groceryrules_eclat <- ruleInduction(groceryitemsets_eclat, confidence = 0.25)
groceryrules_eclat
inspect(groceryrules_eclat[1:5])
